unit IWExtCtrls32;

interface

uses
  {$IFDEF VSNET}
  System.ComponentModel, System.Drawing, IWNetClasses,
  System.IO,
  System.Drawing.Design, System.Drawing.Imaging,
  {$ELSE}
  Classes,
    {$IFDEF Linux}
    IWCLXComponent, IWCLXClasses,
    {$ELSE}
    IWVCLComponent, IWVCLClasses,
    {$ENDIF}
  {$ENDIF}
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  SysUtils,
  IWBaseControl, IWControl32, IWTypes, IWFileReference, IWHTMLTag, IWGlobal,
  IWRenderContext, IWJpegOptions, IWBaseInterfaces, IWBaseHTMLControl;

type
  TIWCustomRadioGroup32 = class(TIWCustomControl32, IIWInputControl)
  protected
    FItemIndex: Integer;
    FItems: TIWStringList;
    FLayout: TIWRadioGroupLayout;
    //
    procedure OnItemsChange( ASender : TObject );
    procedure SetItems(AValue: TIWStringList);
    procedure SetLayout(const AValue: TIWRadioGroupLayout);
    procedure SetValue(const AValue: string); virtual;
    procedure InitControl; override;
    procedure SetItemIndex(AValue : Integer);
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;

    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    property ItemIndex: integer read FItemIndex write SetItemIndex;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property Items: TIWStringList read FItems write SetItems;
    property Layout: TIWRadioGroupLayout read FLayout write SetLayout;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWRadioGroup32.bmp}
  TIWRadioGroup32 = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWRadioGroup32), 'TIWRadioGroup32.bmp')]
  {$ENDIF}
  TIWRadioGroup32 = class(TIWCustomRadioGroup32)
  published
    property Enabled;
    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    property ItemIndex;
    property Items;
    property Layout;
  end;

  TIWCustomImage32 = class(TIWCustomControl32, IIWSubmitControl)
  protected
    FAltText: string;
    FPicture: TIWPicture;
    FUseSize: Boolean;
    FSubmitParam : String;
    {$IFDEF VSNET}
    FImage: Image;
    {$ENDIF}
    //
    function CanAutoSize(var VNewWidth: Integer;
      var VNewHeight: Integer): Boolean; {$IFNDEF Linux} override; {$ENDIF}
    procedure CreatePicture;
    // Necessary because Borland made FOnClick protected
    function ImageHTML(const ASrc: string): TIWHTMLTag;
    procedure PictureChanged(ASender: TObject);
    procedure SetPicture(AValue: TIWPicture);
    function get_HasTabOrder: boolean; override;
    {$IFDEF VSNET}
    procedure SetImage(AValue: Image);
    {$ENDIF}
    procedure InitControl; override;
    function GetPicture: TIWPicture;

    procedure Submit(const AValue: string); override;

    {$IFDEF VSNET}
    function ShouldSerializeImage: Boolean;
    [Bindable(true)]
    property Image: Image read FImage write SetImage;
    {$ENDIF}
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function GetSubmitParam : String;
    property Picture: TIWPicture read GetPicture write SetPicture;
  published
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property AltText: string read FAltText write FAltText;
    {$IFNDEF Linux}
    // Necessary as Delphi defaults to False and causes streaming problems

    property AutoSize default True;
    {$ENDIF}
    property ExtraTagParams;

    property UseSize: Boolean read FUseSize write FUseSize;
    property OnClick; // : TNotifyEvent read GetOnClick write SetOnClick;
  end;

  TIWDynamicImage32 = class(TIWCustomImage32)
  protected
    FJpegOptions: TIWJpegOptions;
    //
    procedure SetJpegOptions(const AValue: TIWJpegOptions);
    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published

    property JpegOptions: TIWJpegOptions read FJpegOptions write SetJpegOptions;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWImage32.bmp}
  TIWImage32 = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWImage32), 'TIWImage32.bmp')]
  {$ENDIF}
  TIWImage32 = class(TIWDynamicImage32)
  published
    {$IFDEF VSNET}
    property Image;
    {$ELSE}
    property Picture;
    {$ENDIF}
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWImageFile32.bmp}
  TIWImageFile32 = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWImageFile32), 'TIWImageFile32.bmp')]
  {$ENDIF}
  TIWImageFile32 = class(TIWCustomImage32)
  protected
    FCacheable: Boolean;
    FImageFile: TIWFileReference;
    //
    procedure ReferenceChanged(ASender: TObject);
    procedure SetImageFile(const AValue: TIWFileReference);

    procedure InitDesignTime; override;
    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    property Cacheable: Boolean read FCacheable write FCacheable;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    [Bindable(true)]    
    {$ENDIF}
    property ImageFile: TIWFileReference read FImageFile write SetImageFile;
  end;

implementation

uses
  {$IFDEF VSNET}
  Classes,
  {$ENDIF}
  IWServerControllerBase, SWSystem;

{ TIWCustomRadioGroup32 }

procedure TIWCustomRadioGroup32.InitControl;
begin
  inherited;
  FItems := TIWStringList.Create;
  FItems.OnChange := OnItemsChange;
  FNeedsFormTag := True;
  Width := 89;
  Height := 112;
end;

procedure TIWCustomRadioGroup32.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FItems);
  inherited;
end;

function TIWCustomRadioGroup32.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  i: Integer;
begin
  // To identify them as a group, HTML requires they all have the same name
  Result := TIWHTMLTag.CreateTag('DIV'); try
    for i := 0 to Items.Count - 1 do begin
      with Result.Contents.AddTag('INPUT') do begin
        AddStringParam('TYPE', 'RADIO');
        Add(IIF(ItemIndex = i, ' CHECKED'));
        AddStringParam('NAME', HTMLName);
        AddStringParam('VALUE', IntToStr(i));
        Add(IIF(not Enabled, 'DISABLED'));
      end;
      Result.Contents.AddTagAsObject(WebFont.FontToStringTag32(Items.Strings[i]));
      Result.Contents.AddText(iif(Layout = glVertical, '<BR>', ' '));
    end;
  except FreeAndNil(Result); raise; end;
end;

procedure TIWCustomRadioGroup32.SetItemIndex(AValue : Integer);
begin
  FItemIndex := AValue;
  if FItemIndex < -1 then begin
    FItemIndex := -1;
  end;
end;

procedure TIWCustomRadioGroup32.SetItems(AValue: TIWStringList);
begin
  FItems.Assign(AValue);
  Invalidate;
end;

procedure TIWCustomRadioGroup32.SetLayout(
  const AValue: TIWRadioGroupLayout);
begin
  FLayout := AValue;
  Invalidate;
end;

procedure TIWCustomRadioGroup32.SetValue(const AValue: string);
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  ItemIndex := StrToIntDef(AValue, -1);

  if ItemIndex < -1 then begin
    ItemIndex := -1;
  end; 

  Invalidate;
end;

procedure TIWCustomRadioGroup32.OnItemsChange(ASender: TObject);
begin
  Invalidate;
end;

{ TIWCustomImage32 }

function TIWCustomImage32.get_HasTabOrder: Boolean;
begin
  Result := Assigned(OnClick);
end;

function TIWCustomImage32.GetPicture: TIWPicture;
begin
  if not Assigned(FPicture) then begin
    CreatePicture;
  end;
  result := FPicture;
end;

function TIWCustomImage32.CanAutoSize(var VNewWidth,
  VNewHeight: Integer): Boolean;
begin
  Result := False;
  // FPicture is nil during initial create at design time for TIWImageFile (inherited Create)
  // Seems wierd to check AutoSize esp during streaming, but its required otehrwise during
  // AutoSize = False and streaming at runtime we get witdh and height of 0.
  if (FPicture <> nil) and (IsDesignMode = False) and (AutoSize = True) then begin
    if (FPicture.Width > 0) and (FPicture.Height > 0) then begin
      Result := True;
    end;
  end;
end;

procedure TIWCustomImage32.InitControl;
begin
  inherited;
  AutoSize := True;
  FUseSize := True;
  FNeedsFormTag := True;
  Width := 89;
  Height := 112;
  FAltText := '';
end;

procedure TIWCustomImage32.CreatePicture;
begin
  FPicture := TIWPicture.Create;
  FPicture.OnChange := PictureChanged;
end;

procedure TIWCustomImage32.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FPicture);
  inherited;
end;

procedure TIWCustomImage32.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
  DoClick;
end;

function TIWCustomImage32.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWCustomImage32.ImageHTML(const ASrc: string): TIWHTMLTag;
begin
  if Assigned(OnClick) then begin
    Result := TIWHTMLTag.CreateTag('INPUT');
    try
      Result.AddStringParam('VALUE', TextToHTML(Name));
      Result.AddStringParam('NAME', HTMLName);
      Result.AddStringParam('TYPE', 'image');
      Result.AddStringParam('SRC', ASrc);
    except
      FreeAndNil(Result);
      raise;
    end;
  end else begin
    Result := TIWHTMLTag.CreateTag('IMG');
    try
      Result.AddStringParam('SRC', ASrc);
      Result.AddStringParam('NAME', HTMLName);
      Result.AddStringParam('ALT', AltText);
      if UseSize then begin
        Result.AddIntegerParam('WIDTH', Width);
        Result.AddIntegerParam('HEIGHT', Height);
      end;
    except
      FreeAndNil(Result);
      raise;
    end;
  end;
end;

procedure TIWCustomImage32.PictureChanged(ASender: TObject);
begin
  if AutoSize and (FPicture.Width > 0) and (FPicture.Height > 0) then begin
    SetBounds(Left, Top, FPicture.Width, FPicture.Height);
  end;
  Invalidate;
end;

{$IFDEF VSNET}
function TIWCustomImage32.ShouldSerializeImage: Boolean;
begin
  result := Assigned(FImage);
end;

procedure TIWCustomImage32.SetImage(AValue: Image);
Var
  ms: MemoryStream;
  ms1: TCLRStreamWrapper;
begin
  FImage := AValue;
  if Assigned(FImage) then begin
    ms := MemoryStream.Create; try
      FImage.Save(ms, ImageFormat.Bmp);
      ms1 := TCLRStreamWrapper.Create(ms); try
        FPicture.Graphic := TBitmap.Create;
        ms1.Position := 0;
        FPicture.Graphic.LoadFromStream(ms1);
      finally
        ms1.Free;
      end;
    finally
      ms.Free;
    end;
  end else begin
    FPicture.Graphic := nil;
  end;
end;
{$ENDIF}

procedure TIWCustomImage32.SetPicture(AValue: TIWPicture);
begin
  FPicture.Assign(AValue);
end;

{ TIWDynamicImage32 }

procedure TIWDynamicImage32.InitControl;
begin
  inherited;
  CreatePicture;
  FJpegOptions := IWJpegOptions.TIWJpegOptions.Create;
end;

procedure TIWDynamicImage32.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FJPegOptions);
  inherited;
end;

function TIWDynamicImage32.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LPathname: string;
begin
  Result := nil;
  LPathname := TIWServerControllerBase.NewCacheFile('jpg', true);
  // Is nil if no picture. Ex: Drop a TIWImage on form, dont load an image, run.
  if FPicture.Graphic <> nil then begin
    if not FPicture.Graphic.Empty then begin
      ToJPegFile(FPicture.Graphic, LPathname, FJPegOptions);
      Result := ImageHTML(GServerController.UserCacheURL + ExtractFilename(LPathname));
    end;
  end;
end;

procedure TIWDynamicImage32.SetJpegOptions(const AValue: TIWJpegOptions);
begin
  FJpegOptions.Assign(AValue);
end;

{ TIWImageFile32 }

procedure TIWImageFile32.InitDesignTime;
begin
  inherited;
  if IsDesignMode then begin
    CreatePicture;
  end;
end;

procedure TIWImageFile32.InitControl;
begin
  inherited;
  FCacheable := True;
  FImageFile := TIWFileReference.Create;
  FImageFile.OnChange := ReferenceChanged;
end;

procedure TIWImageFile32.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FImageFile);
  inherited;
end;

procedure TIWImageFile32.ReferenceChanged(ASender: TObject);
begin
  if IsDesignMode then begin
    if FileExists(ImageFile.Filename) then begin
      Picture.LoadFromFile(ImageFile.Filename);
    end else begin
      Picture.Graphic := nil;
    end;
    Invalidate;
  end else begin
    if AutoSize and (FPicture <> nil) then begin
      if (FPicture.Graphic.Width > 0) and (FPicture.Graphic.Height > 0) then begin
        Width := FPicture.Graphic.Width;
        Height := FPicture.Graphic.Height;
      end;
    end;
  end;
end;

function TIWImageFile32.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
  Result := ImageHTML(ImageFile.Location(iif(Cacheable, GServerController.FilesURL
   , GServerController.FilesNCURL)));
end;

procedure TIWImageFile32.SetImageFile(const AValue: TIWFileReference);
begin
  // Assign calls DoChange
  FImageFile.Assign(AValue);
end;

end.
